<?php

namespace App\Models;

use Eloquent as Model;
use Illuminate\Database\Eloquent\SoftDeletes;


use Spatie\Image\Manipulations;
use Spatie\MediaLibrary\HasMedia\HasMedia;
use Spatie\MediaLibrary\HasMedia\HasMediaTrait;
use Spatie\MediaLibrary\Models\Media;

/**
 * Class DiwanRecord
 * @package App\Models
 * @version June 4, 2021, 2:47 pm UTC
 *
 * @property \App\Models\AccountantYear $accountantYear
 * @property \App\Models\User $user
 * @property integer $user_id
 * @property integer $sequence
 * @property integer $accountant_year_id
 * @property integer $year
 * @property string $date
 * @property boolean $type
 * @property string $sender
 * @property string $conclusion
 * @property string $location
 * @property string $param1
 * @property string $param2
 * @property string $param3
 * @property string $param4
 * @property string $param5
 * @property string $param6
 */
class DiwanRecord extends Model implements HasMedia
{

    use HasMediaTrait {
        getFirstMediaUrl as protected getFirstMediaUrlTrait;
    }
    public $table = 'diwan_records';

    const CREATED_AT = 'created_at';
    const UPDATED_AT = 'updated_at';

    protected $appends = [
        'has_media'
    ];

    public $fillable = [
        'user_id',
        'sequence',
        'accountant_year_id',
        'year',
        'date',
        'type',
        'sender',
        'conclusion',
        'location',
        'number',
        'param1',
        'param2',
        'param3',
        'param4',
        'param5',
        'param6'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'id' => 'integer',
        'user_id' => 'integer',
        'sequence' => 'integer',
        'accountant_year_id' => 'integer',
        'year' => 'integer',

        'date' => 'date:Y/m/d',
        'type' => 'integer',
        'sender' => 'string',
        'conclusion' => 'string',
        'location' => 'string',
        'param1' => 'string',
        'param2' => 'string',
        'param3' => 'string',
        'param4' => 'string',
        'param5' => 'string',
        'param6' => 'string'
    ];

    /**
     * Validation rules
     *
     * @var array
     */
    public static $rules = [
        'user_id' => 'required',
        'sequence' => 'required|integer',
        'accountant_year_id' => 'nullable',
        'year' => 'nullable|integer',
        'date' => 'nullable',
        'type' => 'required|boolean',
        'sender' => 'nullable|string|max:191',
        'conclusion' => 'nullable|string|max:191',
        'location' => 'nullable|string|max:191',
        'param1' => 'nullable|string',
        'param2' => 'nullable|string',
        'param3' => 'nullable|string',
        'param4' => 'nullable|string',
        'param5' => 'nullable|string',
        'param6' => 'nullable|string',
        'created_at' => 'nullable',
        'updated_at' => 'nullable'
    ];
    public function registerMediaConversions(Media $media = null)
    {
        $this->addMediaConversion('thumb')
            ->fit(Manipulations::FIT_CROP, 200, 200)
            ->sharpen(10);

        $this->addMediaConversion('icon')
            ->fit(Manipulations::FIT_CROP, 100, 100)
            ->sharpen(10);
    }

    /**
     * to generate media url in case of fallback will
     * return the file type icon
     * @param string $conversion
     * @return string url
     */
    public function getFirstMediaUrl($collectionName = 'default', $conversion = '')
    {
        $url = $this->getFirstMediaUrlTrait($collectionName);
        $array = explode('.', $url);
        $extension = strtolower(end($array));
        if (in_array($extension, config('medialibrary.extensions_has_thumb'))) {
            return asset($this->getFirstMediaUrlTrait($collectionName, $conversion));
        } else {
            return asset(config('medialibrary.icons_folder') . '/' . $extension . '.png');
        }
    }



    /**
     * Add Media to api results
     * @return bool
     */
    public function getHasMediaAttribute()
    {
        return $this->hasMedia('images') ? true : false;
    }
    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     **/
    public function accountantYear()
    {
        return $this->belongsTo(\App\Models\AccountantYear::class, 'accountant_year_id');
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     **/
    public function user()
    {
        return $this->belongsTo(\App\User::class, 'user_id');
    }
}
